<?php


namespace Mnv\Modules\User;

use Mnv\Core\Config;
use Mnv\Core\Locale\I18N;
use Mnv\Models\Users\UserTypes;
use Mnv\Modules\SmsGateway;

use Mnv\Core\Auth\Errors\AuthError;
use Mnv\Core\Auth\Exceptions\UnknownIdException;
use Mnv\Core\Auth\Errors\UserPhoneRequiredError;
use Mnv\Core\Auth\Exceptions\NotLoggedInException;
use Mnv\Core\Auth\Exceptions\PhoneNotPaidException;
use Mnv\Core\Auth\Exceptions\EmailNotPaidException;
use Mnv\Core\Auth\Exceptions\InvalidPhoneException;
use Mnv\Core\Auth\Exceptions\TokenExpiredException;
use Mnv\Core\Auth\Exceptions\InvalidEmailException;
use Mnv\Core\Auth\Exceptions\ResetDisabledException;
use Mnv\Core\Auth\Exceptions\InvalidPasswordException;
use Mnv\Core\Auth\Exceptions\TooManyRequestsException;
use Mnv\Core\Auth\Exceptions\PhoneNotVerifiedException;
use Mnv\Core\Auth\Exceptions\AttemptCancelledException;
use Mnv\Core\Auth\Exceptions\ValidatePasswordException;
use Mnv\Core\Auth\Exceptions\UserAlreadyExistsException;
use Mnv\Core\Auth\Exceptions\DuplicateUsernameException;
use Mnv\Core\Auth\Exceptions\InvalidSelectorTokenPairException;

use Mnv\Core\Database\Throwable\IntegrityConstraintViolationException;

class Individual extends User implements UserInterface
{
    /** @var SmsGateway */
    protected $sms;

    /**
     * Phone constructor.
     */
    public function __construct()
    {
        parent::__construct();
        $this->sms = new SmsGateway(Config::getValue('sms_url'),Config::getValue('sms_login'), Config::getValue('sms_password'));
    }

    /**
     * Проверка на существование номера телефона
     *
     * @param $user
     */
    public function check($user)
    {
        global $userAuth;

        try {
            $user['phone'] = $this->sms->formatPhoneNumber($user['phone']);
            $userData = $userAuth->getCheckUserPhoneNumber($user['phone']);

            if ( !empty($userData) ) {
                if ( (int)$userData['verified'] === 1 ) {
                    $this->response = array('status' => 200, 'color' => 'green', 'action' => 'authPhone',
                        'message' => I18N::locale("Пользователь найден", "Nastroyte to'g'ri sms shlyuz", "Set up the right sms gateway!"),
                        'button' => I18N::locale("Войти", "Tizimga kirish", "Login"),
                    );
                } else {
                    $userAuth->createConfirmationRequestPhone($userData['userId'], $userData['phone'], 60, function ($phone, $confirmCode) {
                        $phone = $this->sms->formatPhoneNumber($phone);

                        $text = I18N::locale(
                            "Ваш код подтверждения на сайте : $confirmCode",
                            "Saytida tasdiqlash kodi: $confirmCode",
                            "Your confirmation code on the website : $confirmCode"
                        );

                        if ( $this->sms->sendSms($phone, $text) ) {
                            $phoneHide = substr_replace($phone,'****',4,13);
                            $this->response = array('status' => 200, 'sms' => true, 'color'=> 'green', 'action' => 'confirmPhone',
                                'message' => I18N::locale(
                                    "На Ваш номер $phoneHide телефона отправлен код подтверждения!",
                                    "Telefoningizning $phoneHide raqamiga tasdiqlash kodi yuborildi!",
                                    "A confirmation code has been sent to your $phoneHide phone number!"
                                )
                            );
                        } else {
                            $this->response = array('status' => 403, 'color'=> 'red',
                                'message' => I18N::locale(
                                    "Настройте правильно смс шлюз!",
                                    "Nastroyte to'g'ri sms shlyuz!",
                                    "Set up the right sms gateway!"
                                )
                            );
                        }
                    });
                }
            }
        } catch (InvalidPhoneException $e) {
            $this->response = array('status' => 200, 'color'=> 'green', 'action' => 'registrationPhone',
                'message' => I18N::locale("Не зарегистрирован!", "Roʻyxatdan oʻtmagan", "Not registered!"),
                'button' => I18N::locale("Зарегистрироваться", "Roʻyxatdan oʻtish", "Register")
            );
        }catch (PhoneNotVerifiedException $e) {
            $this->response = array('status' => 403, 'color'=> 'red',
                'message' => I18N::locale('Номер телефона не был подтвержден', 'Email tasdiqlanmagan', 'Email not verified')
            );
        } catch (AuthError $e) {
            $this->response =  array('status'=> 403, 'color'=> 'red', 'action' => 'registrationPhone', 'message'=> $e->getMessage());
        }
    }

    /**
     * Авторизация
     *
     * @param $user
     */
    public function auth($user)
    {
        global $userAuth;

        try {

            $rememberDuration = (isset($user['remember']) && $user['remember'] == 1) ? (int) (60 * 60 * 24 * 365.25) : null;
            $user['phone'] = $this->sms->formatPhoneNumber($user['phone']);
//            if (Config::getValue('twoFactorAuth') == 1) {
//                /** подтверждение через смс шлюз */
//                $callback = function ($phone, $tokenSms) {
//                    $phone = $this->sms->formatPhoneNumber($phone);
//                    if ($this->sms->sendSms($phone, "Ваш активационный код на сайте: $tokenSms"))  {
//                        $phoneHide = substr_replace($phone,'****',4,13);
//                        $this->response = array('status' => 200, 'sms' => true, 'color'=> 'green',
//                            'message' => I18N::locale("На Ваш номер $phoneHide телефона отправлен код подтверждения!", "$phone telefon raqamingizga tasdiqlash kodi!", "Confirmation code to your phone number $phone!")
//                        );
//                    } else {
//                        $this->response = array('status' => 403, 'color'=> 'red',
//                            'message' => I18N::locale("Настройте правильно смс шлюз!", "Nastroyte to'g'ri sms shlyuz!", "Set up the right sms gateway!")
//                        );
//                    }
//                };
//            } else {
//                $callback = null;
//                $this->response = array('status' => 200, 'color' => 'green',
//                    'message' => I18N::locale("Добро пожаловать!", "Endi siz tizimga kirishingiz mumkin!", "Now you can authorize!")
//                );
//            }

//            $userAuth->loginWithPhone($user['phone'], $user['password'], (int)Config::getValue('twoFactorAuth'), $rememberDuration, $callback);
            $userAuth->loginWithPhone($user['phone'], $user['password'], 0, $rememberDuration, null);

            $this->response = array('status' => 200, 'color' => 'green', 'sms' => false, 'confirm' => false, 'auth' => true,
                'message' => I18N::locale("Добро пожаловать!", "Endi siz tizimga kirishingiz mumkin!", "Now you can authorize!")
            );
        }
        catch (UserPhoneRequiredError $e) {
            $this->response = array('status' => 403, 'color'=> 'red',
                'message' => I18N::locale("Поле номер телефона не должен быть пустым", "Ushbu telefon raqamiga ega foydalanuvchi ro'yxatdan o'tmagan", 'The user with this phone number is not registered')
            );
        }
        catch (InvalidPhoneException $e) {
            $this->response = array('status' => 403, 'color'=> 'red',
                'message' => I18N::locale("Пользователь с таким номером телефона не зарегистрирован", "Ushbu telefon raqamiga ega foydalanuvchi ro'yxatdan o'tmagan", 'The user with this phone number is not registered')
            );
        }
        catch (ValidatePasswordException $e) {
            $this->response = array('status' => 403, 'color'=> 'red',
                'message' => I18N::locale("Пароль не должен быть пустым или кол-во символов не должно быть меньше 3", "Siz noto'g'ri parolni kiritdingiz", 'You entered the wrong password')
            );
        }
        catch (InvalidPasswordException $e) {
            $this->response = array('status' => 403, 'color'=> 'red',
                'message' => I18N::locale("Вы указали не верный пароль", "Siz noto'g'ri parolni kiritdingiz", 'You entered the wrong password')
            );
        }
        catch (UnknownIdException $e) {
            $this->response = array('status' => 403, 'color'=> 'red',
                'message' => I18N::locale("Пользователь не найден", "Siz noto'g'ri parolni kiritdingiz", 'You entered the wrong password')
            );
        }
        catch (PhoneNotVerifiedException $e) {
            $this->response = array('status' => 403, 'color'=> 'red',
                'message' => I18N::locale('Номер телефона не был подтвержден', 'Email tasdiqlanmagan', 'Email not verified')
            );
        }
        catch (TooManyRequestsException $e) {
            $this->response =  array('status'=> 403, 'color'=> 'red',
                'message'=> I18N::locale("Слишком много запросов", "Juda ko'p so'rovlar", 'Too many requests')
            );
        } catch (AttemptCancelledException $e) {
            $this->response =  array('status'=> 403, 'color'=> 'red',
                'message'=> I18N::locale('Попытка отменена', 'Попытка отменена', 'Попытка отменена')
            );
        } catch (IntegrityConstraintViolationException $e) {
            $this->response = array('status' => 403, 'color'=> 'red',
                'message' => I18N::locale('Ваш ip заблокирован из-за нарушений авторизации', 'Email tasdiqlanmagan', 'Email not verified')
            );
        } catch (AuthError $e) {
            $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> $e->getMessage());
        }

    }

    /**
     * Регистрация по номеру телефона
     *
     * @param $user
     */
    public function registration($user)
    {
        global $userAuth;

        if (!empty($user)) {
            if (!empty($user['phone'])) {
                $user['phone'] = $this->sms->formatPhoneNumber($user['phone']);
                try {
                    if ((int)Config::getValue('twoFactorAuth') == 1) {
                        $callback = function ($phone, $confirmCode) {  /** подтверждение через смс шлюз */
                            $phone = $this->sms->formatPhoneNumber($phone);
                            $text = I18N::locale(
                                "Ваш код подтверждения на сайте: $confirmCode",
                                "Saytida tasdiqlash kodi: $confirmCode",
                                "Your website confirmation code: $confirmCode"
                            );

                            if ($this->sms->sendSms($phone, $text))  {
                                $phoneHide = substr_replace($phone,'****',4,13);
                                $this->response = array('status' => 200, 'sms' => true, 'confirm' => true, 'auth' => false, 'color'=> 'green', 'action' => 'confirmSms',
                                    'message' => I18N::locale(
                                        "На Ваш номер $phoneHide телефона отправлен код подтверждения!",
                                        "Telefoningizning $phoneHide raqamiga tasdiqlash kodi yuborildi!",
                                        "A confirmation code has been sent to your $phoneHide phone number!"
                                    )
                                );
                            } else {
                                $this->response = array('status' => 403, 'color'=> 'red',
                                    'message' => I18N::locale("Настройте правильно смс шлюз!", "Nastroyte to'g'ri sms shlyuz!", "Set up the right sms gateway!")
                                );
                            }
                        };
                    } else {
                        $callback = null;
                        $this->response = array('status' => 200, 'color' => 'green',
                            'message' => I18N::locale("Теперь вы можете авторизоваться!", "Endi siz tizimga kirishingiz mumkin!", "Now you can authorize!")
                        );
                    }

                    $userAuth->registerWithUniqueUsername('phone', $user, UserTypes::CUSTOMER, 'phone', (int)Config::getValue('twoFactorAuth'), $callback);

                }  catch (InvalidEmailException $e) {
                    $this->response =  array('status'=> 403, 'color'=> 'red',
                        'message'=> I18N::locale("Неверный номер телефона", "Yaroqsiz elektron pochta manzili", "Invalid email address")
                    );
                } catch (ValidatePasswordException $e) {
                    $this->response = array('status' => 403, 'color'=> 'red',
                        'message' => I18N::locale("Пароль не должен быть пустым или кол-во символов не должно быть меньше 3", "Siz noto'g'ri parolni kiritdingiz", 'You entered the wrong password')
                    );
                } catch (InvalidPasswordException $e) {
                    $this->response =  array('status'=> 403, 'color'=> 'red',
                        'message'=> I18N::locale("Invalid password", "Invalid password", "Invalid password")
                    );
                } catch (UserAlreadyExistsException $e) {
                    $this->response =  array('status'=> 403, 'color'=> 'red',
                        'message'=> I18N::locale("Такой номер телефона уже зарегистрирован", "Bu telefon raqami allaqachon roʻyxatdan oʻtgan", "This phone number is already registered")
                    );
                } catch (DuplicateUsernameException $e) {
                    $this->response =  array('status'=> 403, 'color'=> 'red',
                        'message'=> I18N::locale("Пользователь с таким номером телефона уже существует уже зарегистрирован", "Foydalanuvchi allaqachon mavjud", "User already exists")
                    );
                } catch (TooManyRequestsException $e) {
                    $this->response =  array('status'=> 403, 'color'=> 'red',
                        'message'=> I18N::locale("Слишком много запросов", "Juda ko'p so'rovlar", "Too many requests")
                    );
                } catch (AuthError $e) {
                    $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> $e->getMessage());
                }
            } else {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Поле email не должно быть пустым!", "'Elektron pochta maydoni bo'sh bo'lmasligi kerak!'", "The email field must not be empty!")
                );
            }
        } else {
            $this->response = array('status' => 403, 'color'=> 'red',
                'message'=> I18N::locale("Отправленные данные пусты!", "Yuborilgan ma'lumotlar boʻsh!", "Sent data is empty!"),
            );
        }
    }

    /**
     * Подтверждение регистрации и авторизации через смс
     *
     * @param $user
     */
    public function confirmSms($user)
    {
        global $userAuth;

        if (!empty($user['code'])) {
            try {
                $userAuth->confirmPhoneAndSignIn($user['code']);
                $this->response =  array('status'=> 200, 'color'=> 'green', 'sms' => false, 'confirm' => false, 'auth' => true,
                    'message'=> I18N::locale("Добро пожаловать!", "Xush kelibsiz!", "Welcome!")
                );

            } catch (InvalidSelectorTokenPairException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Не верный код подтверждения!", "Tasdiq kodi noto'g'ri!", "Invalid confirmation code!")
                );
            } catch (UserAlreadyExistsException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Такой номер телефона уже зарегистрирован", "Bu telefon raqami allaqachon roʻyxatdan oʻtgan", "This phone number is already registered")
                );
            } catch (TokenExpiredException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Время истекло", "Vaqt tugadi", "Time is up")
                );
            } catch (TooManyRequestsException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale('Слишком много запросов', 'Juda koʻp soʻrovlar', 'Too many requests')
                );
            } catch (AuthError $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> $e->getMessage());
            }
        } else {
            $this->response =  array('status'=> 403, 'color'=> 'red',
                'message'=> I18N::locale("Отправленные данные пусты!", "Yuborilgan ma'lumotlar bo'sh!", "Sent data is empty!")
            );
        }
    }

    /**
     * Восстановление пароля
     *
     * @param $user
     */
    public function recoveryPhone($user)
    {
        global $userAuth;

        if (!empty($user) && !empty($user['phone'])) {
            $user['phone'] = $this->sms->formatPhoneNumber($user['phone']);
            if ((int)Config::getValue('twoFactorAuth')  == 1) {
                try {
                    $userAuth->forgotPhonePassword($user['phone'], function ($phone, $confirmCode) {
                        $phone = $this->sms->formatPhoneNumber($phone);

                        $text = I18N::locale(
                            "Ваш код подтверждения для восстановление пароля на сайте: $confirmCode",
                            "Saytida parolni tiklash uchun tasdiqlash kodi: $confirmCode",
                            "Your confirmation code for password recovery on the site: $confirmCode"
                        );

                        if ($this->sms->sendSms($phone, $text)) {
                            $phoneHide = substr_replace($phone,'****',4,13);
                            $this->response = array('status' => 200, 'sms' => true, 'confirm' => false, 'color'=> 'green', 'action' => 'recoveryConfirm',
                                'message' => I18N::locale(
                                    "На Ваш номер $phoneHide телефона отправлен код подтверждения!",
                                    "Telefoningizning $phoneHide raqamiga tasdiqlash kodi yuborildi!",
                                    "A confirmation code has been sent to your $phoneHide phone number!"
                                ),
                                'button' => I18N::locale("Подтвердить", "Tasdiqlash", "Confirm"),
                            );
                        } else {
                            $this->response = array('status' => 403, 'color'=> 'red',
                                'message' => I18N::locale("Настройте правильно смс шлюз!", "Nastroyte to'g'ri sms shlyuz!", "Set up the right sms gateway!")
                            );
                        }
                    }, 60);
                }  catch (InvalidPhoneException $e) {
                    $this->response =  array('status'=> 403, 'color'=> 'red',
                        'message'=> I18N::locale("Такой номер телефона не зарегистрирован", "Bunday telefon raqami ro'yxatdan o'tmagan", "This phone number is not registered")
                    );
                } catch (PhoneNotPaidException $e) {
                    $this->response =  array('status'=> 403, 'color'=> 'red',
                        'message'=> I18N::locale('Не была произведена оплата', 'Ne byla prizvedena oplata', 'Payment has not been made')
                    );
                } catch (PhoneNotVerifiedException $e) {
                    $this->response =  array('status'=> 403, 'color'=> 'red',
                        'message'=> I18N::locale('Номер телефона не подтвержден', 'Telefon raqami tasdiqlanmagan', 'Phone number not verified')
                    );
                } catch (ResetDisabledException $e) {
                    $this->response =  array('status'=> 403, 'color'=> 'red',
                        'message'=> I18N::locale('Сброс пароля отключен', 'Parolni tiklash o\'chirilgan', 'Password reset is disabled')
                    );
                } catch (TooManyRequestsException $e) {
                    $this->response =  array('status'=> 403, 'color'=> 'red',
                        'message'=> I18N::locale('Слишком много запросов', 'Juda koʻp soʻrovlar', 'Too many requests')
                    );
                } catch (AuthError $e) {
                    $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> $e->getMessage());
                }
            }

        } else {
            $this->response =  array('status'=> 403, 'color'=> 'red',
                'message'=> I18N::locale('Отправленные данные пусты!', 'Yuborilgan maʼlumotlar boʻsh!', 'Sent data is empty!')
            );
        }
    }

    /**
     * Подтверждение восстановление пароля
     *
     * @param $user
     */
    public function recoveryConfirmPhone($user)
    {
        global $userAuth;

        if (!empty($user['code'])) {
            try {
                $phoneBeforeAndAfter = $userAuth->confirmPhone($user['code']);

                if ($phoneBeforeAndAfter[1] !== null) {
                    $userAuth->recoveryPhonePassword($phoneBeforeAndAfter[1], $user['newPassword'], function ($phone, $password) use ($userAuth) {

                        $rememberDuration = (isset($user['remember']) && $user['remember'] == 1) ? (int) (60 * 60 * 24 * 365.25) : null;
                        $phone = $this->sms->formatPhoneNumber($phone);

                        $userAuth->loginWithPhone($phone, $password, 0, $rememberDuration, null);
                        $this->response = array('status' => 200, 'color' => 'green', 'sms' => false, 'confirm' => true,
                            'message' => I18N::locale("Добро пожаловать!", "Xush kelibsiz!", "Welcome!"),
                            'button' => I18N::locale("Добро пожаловать!", "Xush kelibsiz!", "Welcome!"),
                        );
                    });
                } else {
                    $this->response = array('status' => 200, 'color' => 'green', 'sms' => false, 'confirm' => true,
                        'message' => I18N::locale("Ошибка изменения", "Raqamni o'zgartirish xatosi", "Number change error"),
                        'button' => I18N::locale("Ошибка", "Xato", "Error"),
                    );
                }

            } catch (InvalidSelectorTokenPairException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Не верный код подтверждения!", "Kod noto'g'ri!", "Incorrect code!")
                );
            } catch (UserAlreadyExistsException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Такой номер телефона уже зарегистрирован", "Bu telefon raqami allaqachon roʻyxatdan oʻtgan", "This phone number is already registered")
                );
            } catch (PhoneNotVerifiedException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Ваш номер телефона не подтвержден ", "Foydalanuvchi allaqachon mavjud", "User already exists")
                );
            } catch (InvalidPhoneException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Такой номер телефона не зарегистрирован", "Bunday telefon raqami ro'yxatdan o'tmagan", "This phone number is not registered")
                );
            } catch (ResetDisabledException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Изменение пароля запрещено", "Parolni o'zgartirish taqiqlanadi", "Changing the password is prohibited")
                );
            } catch (TooManyRequestsException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Слишком много запросов", "Juda ko'p soʻrovlar", "Too many requests")
                );
            } catch (TokenExpiredException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Время истекло", "Vaqt tugadi", "Time is up")
                );
            }  catch (AuthError $e) {
                $this->response = array('status'=> 403, 'color'=> 'red', 'message'=> $e->getMessage());
            }
        } else {
            $this->response =  array('status'=> 403, 'color'=> 'red',
                'message'=> I18N::locale("Отправленные данные пусты!", "Yuborilgan ma'lumotlar bo'sh!", "Sent data is empty!")
            );
        }
    }

    /**
     * Изменение номера телефона
     *
     * @param $user
     */
    public function changePhone($user)
    {
        global $userAuth;

        if (!empty($user)) {
            try {
                $user['phone'] = $this->sms->formatPhoneNumber($user['phone']);
                $userAuth->changeConfirmUserPhone($user['phone'], 60, function ($phone, $confirmCode) {
                    $phone = $this->sms->formatPhoneNumber($phone);
                    $text = I18N::locale(
                        "Ваш код подтверждения для изменения номера телефона на сайте: $confirmCode",
                        "Saytdagi telefon raqamingizni o'zgartirish uchun tasdiqlash kodingiz: $confirmCode",
                        "Your confirmation code for changing your phone number on the site: $confirmCode"
                    );
                    if ($this->sms->sendSms($phone, $text)) {
                        $phoneHide = substr_replace($phone,'****',4,13);

                        $this->response = array('status' => 200, 'sms' => true, 'confirm' => false, 'color'=> 'green', 'action' => 'changeConfirmContact',
                            'message' => I18N::locale(
                                "На Ваш номер $phoneHide телефона отправлен код подтверждения!",
                                "Tasdiqlash kodi $phoneHide telefon raqamingizga yuborildi!",
                                "A verification code has been sent to your $phoneHide phone number!"
                            ),
                            'button' => I18N::locale("Подтвердить", "Tasdiqlash", "Confirm"),
                        );

                    } else {
                        $this->response = array('status' => 403, 'color'=> 'red',
                            'message' => I18N::locale("Настройте правильно смс шлюз!", "Nastroyte to'g'ri sms shlyuz!", "Set up the right sms gateway!")
                        );
                    }
                });
            }  catch (NotLoggedInException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Вы не авторизованы", "Sizga ruxsat berilmagan", "You are not authorized")
                );
            } catch (UserAlreadyExistsException $e) {
                $this->response =  ['status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Такой номер телефона уже зарегистрирован", "Bu telefon raqami allaqachon roʻyxatdan oʻtgan", "This phone number is already registered")
                ];
            } catch (TooManyRequestsException $e) {
                $this->response =  ['status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Слишком много запросов!", "Too many requests!", "Too many requests!")
                ];
            }
        } else {
            $this->response =  array('status'=> 403, 'color'=> 'red',
                'message'=> I18N::locale("Отправленные данные пусты!", "Yuborilgan ma'lumotlar bo'sh!", "Sent data is empty!")
            );
        }
    }

    /**
     * Подтверждение на изменение номера телефона
     *
     * @param $user
     */
    public function changeConfirmContact($user)
    {
        global $userAuth;

        if (!empty($user['code'])) {
            try {
                $phoneBeforeAndAfter = $userAuth->confirmPhone($user['code']);
                if ($phoneBeforeAndAfter[1] !== null) {
                    $this->response = array('status' => 200, 'color' => 'green', 'sms' => false, 'confirm' => true,
                        'message' => I18N::locale("Ваш номер телефона успешно изменен", "Telefon raqamingiz muvaffaqiyatli o'zgartirildi", "Your phone number has been successfully changed"),
                        'button' => I18N::locale("Изменить", "O'zgartirish", "Change"),
                    );
                } else {
                    $this->response = array('status' => 200, 'color' => 'green', 'sms' => false, 'confirm' => true,
                        'message' => I18N::locale("Ошибка изменения номера", "Raqamni o'zgartirish xatosi", "Number change error"),
                        'button' => I18N::locale("Ошибка", "Xato", "Error"),
                    );
                }

            } catch (InvalidSelectorTokenPairException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Не верный код подтверждения!", "Tasdiq kodi noto'g'ri!", "Invalid confirmation code!")
                );
            } catch (UserAlreadyExistsException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Такой номер телефона уже зарегистрирован", "Bu telefon raqami allaqachon roʻyxatdan oʻtgan", "This phone number is already registered")
                );
            }  catch (TokenExpiredException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Время истекло", "Vaqt tugadi", "Time is up")
                );
            } catch (TooManyRequestsException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Слишком много запросов", "Juda ko'p soʻrovlar", "Too many requests")
                );
            } catch (AuthError $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> $e->getMessage());
            }
        } else {
            $this->response =  array('status'=> 403, 'color'=> 'red',
                'message'=> I18N::locale("Отправленные данные пусты!", "Yuborilgan ma'lumotlar bo'sh!", "Sent data is empty!")
            );
        }

    }

    /**
     * Добавление/Изменение паспортных данных
     *
     * @param $user
     */
    public function changePassport($user)
    {
        global $userAuth;

        if (!empty($user)) {
            try {
                $userAuth->changeUserPassport($user);
                $this->response = array('status' => 200, 'color' => 'green',
                    'message' => I18N::locale("Ваши данные успешно изменены", "Sizning ma'lumotlar muvaffaqiyatli o'zgardi", "Your data has been successfully changed")
                );
            } catch (NotLoggedInException $e) {
                $this->response = array('status' => 403, 'color' => 'red',
                    'message'=> I18N::locale("Вы не авторизованы", "Sizga ruxsat berilmagan", "You are not authorized")
                );
            } catch (TooManyRequestsException $e) {
                $this->response = ['status' => 403, 'color' => 'red',
                    'message' => I18N::locale("Слишком много запросов!", "Too many requests!", "Too many requests!")
                ];
            }
        } else {
            $this->response =  array(
                'status'=> 403, 'color'=> 'red',
                'message'=> I18N::locale("Отправленные данные пусты!", "Yuborilgan ma'lumotlar bo'sh!", "Sent data is empty!")
            );
        }
    }

    //    /**
//     * @param $user
//     * @throws \Mnv\App\Auth\Errors\DatabaseError
//     */
//    public function changePhone($user)
//    {
//        global $userAuth;
//
//        if (!empty($user)) {
//            $user['phone'] = $this->sms->formatPhoneNumber($user['phone']);
//            try {
//                $userAuth->changeUserPhone($user);
//                $this->response =  array(
//                    'status'=> 200, 'color'=> 'green',
//                    'message'=> I18N::locale("Ваши данные успешно изменены", "Password has been changed", "Password has been changed")
//                );
//            }  catch (NotLoggedInException $e) {
//                $this->response =  array(
//                    'status'=> 403, 'color'=> 'red',
//                    'message'=> I18N::locale("Not logged in", "Not logged in", "Not logged in")
//                );
//            } catch (TooManyRequestsException $e) {
//                $this->response =  [
//                    'status'=> 403, 'color'=> 'red',
//                    'message'=> I18N::locale("Слишком много запросов!", "Too many requests!", "Too many requests!")
//                ];
//            }
//        } else {
//            $this->response =  array(
//                'status'=> 403, 'color'=> 'red',
//                'message'=> I18N::locale("Отправленные данные пусты!", "Yuborilgan ma'lumotlar bo'sh!", "Sent data is empty!")
//            );
//        }
//    }

}